/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef SERIAL_V2_H
#define SERIAL_V2_H
#include <QThread>
#include <QtSerialPort/QSerialPort>
#include <QSerialPortInfo>
#include <iostream>
#include <math.h>
#include "gpioclassv2.h"


//CORE Functions
void	serial_Setup(QSerialPort& serial, QString portname);
void	serialClearBuffer(QSerialPort& serial);
QString	serialWriteRead(QSerialPort& serial, QString tx, bool output_to_console = true);
void	serialRead(QSerialPort& serial, QString& rx, int& message_fail_counter, int& message_fail_limit);
void	serialWrite(QSerialPort& serial, QString tx, bool output_to_console);

QList<QSerialPortInfo>	serial_get_SGX_port_quiet();	//Doesn't generate messages and doesn't exit application
//SYSTEM COMMANDS
QString	serial_setChannel(QSerialPort& serial, QString channel);
QString serial_reset(QSerialPort& serial, QString channel);

QString	serial_set_clock_source(QSerialPort& serial, QString channel, int clock_source);
QString	serial_get_clock_source(QSerialPort& serial, QString channel);
QString	serial_set_PA_type(QSerialPort& serial, QString channel, int PA_type);
QString	serial_get_PA_type(QSerialPort& serial, QString channel);

QString	serial_setIdle_enable(QSerialPort& serial, QString channel, bool input);
QString	serial_getIdentity(QSerialPort& serial, QString channel);
QString	serial_getStatus(QSerialPort& serial, QString channel, bool console_output_enabled = true);
QString	serial_clearError(QSerialPort& serial, QString channel, bool console_output_enabled = true);
QString	serial_getVersion(QSerialPort& serial, QString channel);

QString serial_setAutoGain_enable(QSerialPort& serial, QString channel, bool input);
QString serial_setAnalogInput(QSerialPort& serial, QString channel, bool enable, int modulator_select, double power_min, int ADC_min, double power_max, double ADC_max);

//BASIC COMMANDS
QString	serial_setRF_enable(QSerialPort& serial, QString channel, bool input);
QString	serial_getRF_status(QSerialPort& serial, QString channel, bool console_output_enabled = true);
QString	serial_getTemperature(QSerialPort& serial, QString channel, bool console_output_enabled = true);
void	serial_readPower_dbm(QSerialPort& serial, QString channel, double& frw, double& rfl, bool console_output_enabled);
void	serial_readPower_watt(QSerialPort& serial, QString channel, double& frw, double& rfl, bool console_output_enabled);

//CW
QString	serial_setFrequency(QSerialPort& serial, QString channel, double input_freq_MHz);
QString	serial_getFrequency(QSerialPort& serial, QString channel, bool console_output_enabled = true);
QString	serial_setPower_watt(QSerialPort& serial, QString channel, double input_power_watt);
QString	serial_getPower_watt(QSerialPort& serial, QString channel);
QString	serial_setPower_dbm(QSerialPort& serial, QString channel, double input_power_dbm);
QString	serial_getPower_dbm(QSerialPort& serial, QString channel);
QString	serial_setPhase(QSerialPort& serial, QString channel, double input_phase);
QString	serial_getPhase(QSerialPort& serial, QString channel);

QString serial_setSGxPower_dbm(QSerialPort& serial, QString channel, double sgx_power_dbm);
QString serial_getSGxPower_dbm(QSerialPort& serial, QString channel);
QString serial_setMagnitude(QSerialPort& serial, QString channel, double magnitude);
QString serial_getMagnitude(QSerialPort& serial, QString channel, bool console_output_enabled);
QString serial_setVGA_attenuation(QSerialPort& serial, QString channel, double attenuation_value);
QString serial_getVGA_attenuation(QSerialPort& serial, QString channel, bool console_output_enabled);

//PWM
QString	serial_setPWM_enable(QSerialPort& serial, QString channel, bool input, double input_percentage);
QString	serial_setPWM_freq(QSerialPort& serial, QString channel, double input_freq_Hz, double correction_factor, double delay);
QString	serial_setPWM_DC(QSerialPort& serial, QString channel, double input_percentage);
QString	serial_setPWM_triggering(QSerialPort& serial, QString channel, int PWM_trigger_mode, int slave_port, int slave_pin, int master_port, int master_pin, int trigger_pulse = 0, int refresh_rate = 0);
QString	serial_getPWM_settings(QSerialPort& serial, QString channel);

//DLL
QString	serial_setDLL_enable(QSerialPort& serial, QString channel, bool input);
QString	serial_setDLL_settings(QSerialPort& serial, QString channel, double DLL_freq_low, double DLL_freq_high, double DLL_start_freq, double DLL_step_freq, double DLL_threshold, int DLL_main_delay);
QString	serial_getDLL_settings(QSerialPort& serial, QString channel);
QString	serial_getDLL_status(QSerialPort& serial, QString channel);

//ALL
QString	serial_setALL_enable(QSerialPort& serial, QString channel, bool input);
QString	serial_setALL_settings(QSerialPort& serial, QString channel, double lowerFreq, double upperFreq, double threshold);
QString	serial_getALL_settings(QSerialPort& serial, QString channel);
QString	serial_getALL_status(QSerialPort& serial, QString channel);

QString	serial_set_DVGA_forward(QSerialPort& serial, QString channel, bool input, double attenuation);
QString	serial_set_DVGA_reflected(QSerialPort& serial, QString channel, bool input, double attenuation);
QString	serial_get_DVGA_forward(QSerialPort& serial, QString channel);
QString	serial_get_DVGA_reflected(QSerialPort& serial, QString channel);

//RF Splitter
QString serial_setRF_splitter_channels_connect(QSerialPort& serial, QString channel, int enable_bitmask);

//PSU
QString serial_setPSU_enable(QSerialPort& serial, QString channel, bool input);
QString serial_getPSU_enable(QSerialPort& serial, QString channel);
QString serial_setPSU_voltage_setpoint(QSerialPort& serial, QString channel, double voltage_setpoint);
QString	serial_getPSU_voltage_setpoint(QSerialPort& serial, QString channel);
QString serial_setPSU_current_limit(QSerialPort& serial, QString channel, double current_limit);
QString	serial_getPSU_current_limit(QSerialPort& serial, QString channel);
QString serial_getPSU_IU_All_Reading(QSerialPort& serial, QString channel, bool console_output_enabled = true);


//SWEEP
QString	serial_runSweep_watt(QSerialPort& serial, QString channel, double swpStartFrequency, double swpStopFrequency, double swpStepFrequency, double swpPower_watt);
QString	serial_runSweep_dbm(QSerialPort& serial, QString channel, double swpStartFrequency, double swpStopFrequency, double swpStepFrequency, double swpPower_dBm);


//DEBUG
QString serial_Test(QSerialPort& serial, QString channel, int count, bool console_output_enabled = true);	//TEST function of the firmware. Functionality prone to change any time.

//transparent mode
#ifdef ROTARY_ENCODER_ENABLED
void	serial_RCM_Receive(QSerialPort& input_UART, QSerialPort& output_UART, int &change_indicator, double &frequency, bool &rf_status, double &power_dbm, double &power_watt);
void	serial_RCM_Return(QSerialPort& input_UART, QSerialPort& output_UART, int &change_indicator, double &frw, double &rfl, double &temperature);
#endif
#ifndef ROTARY_ENCODER_ENABLED
void	serial_RCM_Transmit(QIODevice& input_UART, QSerialPort& output_UART);
void	serial_RCM_Receive(QIODevice& input_UART, QSerialPort& output_UART);

#if defined (Q_OS_LINUX)
void	serial_RCM_Receive_RS485(QIODevice& input_UART, QSerialPort& output_UART, GPIOClass& gpio_flipper);
#endif
#endif
#endif // SERIAL_V2_H
